#!/bin/sh

# Copyright © 2022-2025 Audinate Pty Ltd ACN 120 828 006 (Audinate). All rights reserved.
#
#
# 1.	Subject to the terms and conditions of this Licence, Audinate hereby grants you a worldwide, non-exclusive,
#		no-charge, royalty free licence to copy, modify, merge, publish, redistribute, sublicense, and/or sell the
#		Software, provided always that the following conditions are met:
#		1.1.	the Software must accompany, or be incorporated in a licensed Audinate product, solution or offering
#				or be used in a product, solution or offering which requires the use of another licensed Audinate
#				product, solution or offering. The Software is not for use as a standalone product without any
#				reference to Audinate's products;
#		1.2.	the Software is provided as part of example code and as guidance material only without any warranty
#				or expectation of performance, compatibility, support, updates or security; and
#		1.3.	the above copyright notice and this License must be included in all copies or substantial portions
#				of the Software, and all derivative works of the Software, unless the copies or derivative works are
#				solely in the form of machine-executable object code generated by the source language processor.
#
# 2.	TO THE EXTENT PERMITTED BY APPLICABLE LAW, THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
#		EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
#		PURPOSE AND NONINFRINGEMENT.
#
# 3.	TO THE FULLEST EXTENT PERMITTED BY APPLICABLE LAW, IN NO EVENT SHALL AUDINATE BE LIABLE ON ANY LEGAL THEORY
#		(INCLUDING, WITHOUT LIMITATION, IN AN ACTION FOR BREACH OF CONTRACT, NEGLIGENCE OR OTHERWISE) FOR ANY CLAIM,
#		LOSS, DAMAGES OR OTHER LIABILITY HOWSOEVER INCURRED.  WITHOUT LIMITING THE SCOPE OF THE PREVIOUS SENTENCE THE
#		EXCLUSION OF LIABILITY SHALL INCLUDE: LOSS OF PRODUCTION OR OPERATION TIME, LOSS, DAMAGE OR CORRUPTION OF
#		DATA OR RECORDS; OR LOSS OF ANTICIPATED SAVINGS, OPPORTUNITY, REVENUE, PROFIT OR GOODWILL, OR OTHER ECONOMIC
#		LOSS; OR ANY SPECIAL, INCIDENTAL, INDIRECT, CONSEQUENTIAL, PUNITIVE OR EXEMPLARY DAMAGES, ARISING OUT OF OR
#		IN CONNECTION WITH THIS AGREEMENT, ACCESS OF THE SOFTWARE OR ANY OTHER DEALINGS WITH THE SOFTWARE, EVEN IF
#		AUDINATE HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH CLAIM, LOSS, DAMAGES OR OTHER LIABILITY.
#
# 4.	APPLICABLE LEGISLATION SUCH AS THE AUSTRALIAN CONSUMER LAW MAY APPLY REPRESENTATIONS, WARRANTIES, OR CONDITIONS,
#		OR IMPOSES OBLIGATIONS OR LIABILITY ON AUDINATE THAT CANNOT BE EXCLUDED, RESTRICTED OR MODIFIED TO THE FULL
#		EXTENT SET OUT IN THE EXPRESS TERMS OF THIS CLAUSE ABOVE "CONSUMER GUARANTEES".	 TO THE EXTENT THAT SUCH CONSUMER
#		GUARANTEES CONTINUE TO APPLY, THEN TO THE FULL EXTENT PERMITTED BY THE APPLICABLE LEGISLATION, THE LIABILITY OF
#		AUDINATE UNDER THE RELEVANT CONSUMER GUARANTEE IS LIMITED (WHERE PERMITTED AT AUDINATE'S OPTION) TO ONE OF
#		FOLLOWING REMEDIES OR SUBSTANTIALLY EQUIVALENT REMEDIES:
#		4.1.	THE REPLACEMENT OF THE SOFTWARE, THE SUPPLY OF EQUIVALENT SOFTWARE, OR SUPPLYING RELEVANT SERVICES AGAIN;
#		4.2.	THE REPAIR OF THE SOFTWARE;
#		4.3.	THE PAYMENT OF THE COST OF REPLACING THE SOFTWARE, OF ACQUIRING EQUIVALENT SOFTWARE, HAVING THE RELEVANT
#				SERVICES SUPPLIED AGAIN, OR HAVING THE SOFTWARE REPAIRED.
#
# 5.	This License does not grant any permissions or rights to use the trade marks (whether registered or unregistered),
#		the trade names, or product names of Audinate.
#
# 6.	If you choose to redistribute or sell the Software you may elect to offer support, maintenance, warranties,
#		indemnities or other liability obligations or rights consistent with this License. However, you may only act on
#		your own behalf and must not bind Audinate. You agree to indemnify and hold harmless Audinate, and its affiliates
#		from any liability claimed or incurred by reason of your offering or accepting any additional warranty or additional
#		liability.
#

# shellcheck disable=SC1091
. ./dep_util.sh

if [ -f ./dep_check_platform.sh ]; then 
    . ./dep_check_platform.sh
fi

# variables 
EXIT_VALUE=0
ROOTUID="0"
USER=$(whoami)
PATH_TO_CONFIG_FILES="$1"/dante_package/dante_data/capability
PATH_TO_BUNDLE_DIR="$1"/dante_package/bundle
DANTE_JSON="$PATH_TO_CONFIG_FILES"/dante.json
CONFIG_JSON="$PATH_TO_CONFIG_FILES"/config.json
KERNEL_CONFIG=$(kernel_info)
NUM_CORES=$(nproc --all)
MAX_CORE_ID=$((NUM_CORES-1))
CGROUPS="memory devices"
CGROUP_ERR=0
NUM_DEP_CORES=
FOUND_IFACES=
INTERFACES=
HW_INTERFACES=
INTERFACE_MODE=
DSA_TAGGED_PACKETS=
ENCODING_VALID=

# error/warning/notice messages
# NOTE: do not "fix" the alignment of the following variables as their content will be
# put as it is on stdout by echo
CGROUP_ERROR_MSG="One or more cgroup mounts were not found: automatic cgroup resource allocation
            needs to be disabled by setting numDepCores to 0 in dante.json."
CGROUP_WARNING_MSG="One or more cgroup mounts were not found: DEP will only run with numDepCores = 0 in dante.json."
HW_TIMESTAMP_ERR_MSG="Hardware timestamping cannot be enabled when using this interface.
            NOTE: when 'dsaTaggedPackets' is set to true, the interface used for hardware
            timestamping needs to be specified in 'clock.hardwareInterfaces'"
HW_TIMESTAMP_AES67_PTPV1_ERR_MSG="AES67 cannot be supported with PTPv1 timestamping
            To support AES67, enableHwTimestamping must be set to either true or false"
HW_TIMESTAMP_PTPV1_OPTION_INFO_MSG="The selected interface(s) support(s) timestamping of PTPv1 layer 4 event RX packets (HWTSTAMP_FILTER_PTPV1_L4_EVENT)
            enableHwTimestamping can be set to \"v1\" IF unicast domain clocking is not required"
HW_TIMESTAMP_ALL_OPTION_INFO_MSG="The selected interface(s) support(s) timestamping of all RX packets (HWTSTAMP_FILTER_ALL)
            enableHwTimestamping should be set to true instead"
print_device_config_message() {
    echo "            NOTE: '$1' major number is $2 and this needs to be declared under "
    echo "            linux.resources.devices and linux.devices in config.json."
    echo "            Please look at the provided config.json for some examples."
}

print_timestamp_suggestion() {
    echo "            It is recommended to enable hardware timestamping in dante.json for better clocking stability."
    echo "            To do so, set 'enableHwTimestamping' to true in dante.json."
}


## utilities not sourced from dep_util.sh

## functions

fail() { exit 1; }
success() { exit 0; }

# check if USER is root, otherwise exit
check_uid() {
    if [ "$(id -u)" -ne "$ROOTUID" ] ; then
        logerr "this script must be executed with root privileges (use sudo)"
        fail
    fi
}

# check that:
# - at least one argument is provided
# - it is a full path
# - it exists
check_arg() {
    if [ -z "$1" ]; then
        logerr "missing argument: DEP installation path"
        loginfo "usage:"
        loginfo "$(TAB)sudo ./dep_check.sh <DEP installation path>"
        fail
    fi

    if [ ! -d "$1" ]; then
        logerr "$1 is not a valid path"
        fail
    fi
}

check_util() {
    if [ -z "$(which "$1")" ]; then
        logerr "$1 not found"
        fail
    fi
}

# check we are running as root
check_uid

# make sure the first arg is a valid path
check_arg "$1"

# make sure we have gunzip
check_util gunzip 

check_architecture() {
    loginfo "checking DEP and Host architecture..."

    command_available hexdump
    if [ "$RETVAL" = "absent" ]; then
        logwarn "Host architecuture check skipped: 'hexdump' command not found"
        return
    fi

    CRUN_BINARY="crun"
    HOST_BINARY=$(which uname)
    ELF_ISA_OFFSET=0x12
    ELF_ISA_SIZE=2
    ELF_ISA_ARM_32=28
    ELF_ISA_X86_64=3e
    ELF_ISA_ARM_64=b7
    # get the hexdecimal value of the architecture in the ELF header section
    HOST_ARCH=$(hexdump -e '"%x"' -s $ELF_ISA_OFFSET -n $ELF_ISA_SIZE $HOST_BINARY)
    DEP_ARCH=$(hexdump -e '"%x"' -s $ELF_ISA_OFFSET -n $ELF_ISA_SIZE $CRUN_BINARY)

    DEP_ARCH_NAME=
    case $DEP_ARCH in
        "$ELF_ISA_ARM_32" )
            DEP_ARCH_NAME="arm32";;
        "$ELF_ISA_X86_64" )
            DEP_ARCH_NAME="x86_64";;
        "$ELF_ISA_ARM_64" )
            DEP_ARCH_NAME="arm64";;
        * )
            DEP_ARCH_NAME="ELF_ISA: $DEP_ARCH";;
    esac

    HOST_ARCH_NAME=
    case $HOST_ARCH in
        "$ELF_ISA_ARM_32" )
            HOST_ARCH_NAME="arm32";;
        "$ELF_ISA_X86_64" )
            HOST_ARCH_NAME="x86_64";;
        "$ELF_ISA_ARM_64" )
            HOST_ARCH_NAME="arm64";;
        * )
            HOST_ARCH_NAME="ELF_ISA: $HOST_ARCH";;
    esac

    if [ "$HOST_ARCH" = "$DEP_ARCH" ]; then
        logok "DEP and Host architecture matched"
        return
    else
        if [ "$HOST_ARCH" = "$ELF_ISA_ARM_64" ] && [ "$DEP_ARCH" = "$ELF_ISA_ARM_32" ]; then
            # execute crun without arguments to check whether the host is available for multiple architecture supports
            # suppressing both output and error messages of crun
            ./${CRUN_BINARY} > /dev/null 2>&1
            CHECK_MULTI_ARCH_SUPPORT=$?
            if [ "$CHECK_MULTI_ARCH_SUPPORT" -eq 0 ]; then
                logwarn "Running ARM32 DEP on an ARM64 Host. Recommend installing the ARM64 DEP instead"
                return
            else
                logerr "wrong DEP architecture detected: Host CPU is $HOST_ARCH_NAME while DEP was built for $DEP_ARCH_NAME"
                EXIT_VALUE=1
                return
            fi
        else
            logerr "wrong DEP architecture detected: Host CPU is $HOST_ARCH_NAME while DEP was built for $DEP_ARCH_NAME"
            EXIT_VALUE=1
            return
        fi
    fi
}

check_kernel_config() {

    if [ -z "$KERNEL_CONFIG" ]; then
        logerr "kernel config not found, check failed"
        EXIT_VALUE=1
        return 0
    fi

    inner_check_kernel_config() {
        ENABLED=$(echo "$KERNEL_CONFIG" | grep -c "$CONFIG_OPTION"=y)
        MODULE=$(echo "$KERNEL_CONFIG" | grep -c "$CONFIG_OPTION"=m)

        if [ "$1" = "disabled" ]; then # check the kernel configs which should be disabled 
            if [ "$ENABLED" -ne 0 ] || [ "$MODULE" -ne 0 ]; then
                logerr "$CONFIG_OPTION needs to be disabled in kernel config!"
                EXIT_VALUE=1
            else
                logok "$CONFIG_OPTION already disabled"
            fi
            return
        fi

        if [ "$ENABLED" -eq 1 ]; then
            logok "$CONFIG_OPTION found"
            return
        fi

        if [ "$MODULE" -eq 1 ]; then
            logok "$CONFIG_OPTION found as module"
            return
        fi

        if [ "$1" = "required" ]; then
            logerr "$CONFIG_OPTION not in kernel config"
            EXIT_VALUE=1
        else
            logwarn "$CONFIG_OPTION not in kernel config"
        fi
    }

    COMMON_REQUIRED_KCONFIGS="CGROUPS CGROUP_DEVICE CPUSETS CGROUP_SCHED MEMCG NAMESPACES UTS_NS IPC_NS USER_NS PID_NS NET_NS SQUASHFS SQUASHFS_ZLIB BLK_DEV_LOOP IP_MULTICAST"

    loginfo "checking required kernel configs..."
    for OPTION in $COMMON_REQUIRED_KCONFIGS $PLATFORM_REQUIRED_KCONFIGS
    do
        CONFIG_OPTION="CONFIG_""$OPTION"
        inner_check_kernel_config "required"
    done

    loginfo "checking recommended kernel configs..."

    CONFIG_OPTION="CONFIG_PREEMPT"
    inner_check_kernel_config

    # DEP doesn't support fine grained real time scheduling per cgroup yet.
    # Hence, CONFIG_RT_GROUP_SCHED needs to be disabled to start DEP
    CONFIG_OPTION="CONFIG_RT_GROUP_SCHED"
    inner_check_kernel_config "disabled"
}


check_cgroups() {
    display_warning() { logwarn "cgroup mount '$1' not found where expected"; }
    display_found() { logok "cgroup mount '$1' found"; }

    loginfo "checking cgroup mounts..."

    # cpu
    RETVAL=$(mount | grep -c -e "cgroup on /sys/fs/cgroup/cpu " -e "cgroup on /sys/fs/cgroup/cpu,")
    if [ "$RETVAL" -eq 1 ]; then
        display_found "cpu"
    else
        display_warning "cpu"
        CGROUP_ERR=1
    fi

    # check for the cpuset mount if this is a multi-core system
    if [ "$NUM_CORES" -gt 1 ]; then 
        CGROUPS="cpuset "$CGROUPS
    fi

    for CGROUP in $CGROUPS
    do
        RETVAL=$(mount -t cgroup | grep -c "/sys/fs/cgroup/$CGROUP ")
        if [ "$RETVAL" -eq 1 ]; then
            display_found "$CGROUP"
            continue
        fi

        display_warning "$CGROUP"
        CGROUP_ERR=1
    done

    # display suggestion for warnings
    if [ "$CGROUP_ERR" -eq 1 ]; then
        if [ "$NUM_DEP_CORES" -ne 0 ]; then
            logerr "$CGROUP_ERROR_MSG"
        else
            logwarn "$CGROUP_WARNING_MSG"
        fi
    fi
}


check_rng_speed() {
    loginfo "checking random number generator speed..."

    RUN_RNG_SPEED_TEST=1
    command_available /usr/bin/time
    if [ "$RETVAL" = "absent" ]; then
        logwarn "'/usr/bin/time' command not found"
        RUN_RNG_SPEED_TEST=0
    fi

    command_available dd
    if [ "$RETVAL" = "absent" ]; then
        logwarn "'dd' command not found"
        RUN_RNG_SPEED_TEST=0
    fi

    if [ "$RUN_RNG_SPEED_TEST" -eq 1 ]; then 
        
        # try to use more basic system utilities such as ps and sleep rather than timeout 
        dd if=/dev/random of=/dev/null bs=1024 count=1 iflag=fullblock >/tmp/dep_check_out 2>&1 &
        PID=$!
        sleep 1
        kill $PID > /dev/null 2> /dev/null
        wait $PID > /dev/null 2> /dev/null
        RETVAL=$?
        OUTPUT=$(cat /tmp/dep_check_out)
        if [ $RETVAL -ne 0 ]; then # dd did not complete successfully
            if [ -n "$OUTPUT" ]; then # some output was produced, dd probably had an error
                logerr "$OUTPUT"
            else # dd did not produce output
                logerr "dd was killed before producing any output. /dev/random is probably too slow"
            fi
            EXIT_VALUE=1
            return
        fi # dd did not time out

        SPEED_TEST_OUTPUT=$({ /usr/bin/time -p dd if=/dev/random of=/dev/null bs=1024 count=1 iflag=fullblock; } 2>&1 | grep real)
        # At this stage, SPEED_TEST_OUTPUT will have a form like "real 0.00". Note that we use the GNU time here which only
        # contains two decimal point precision. This does not mean "0.00" took 0 seconds to finish but rather was less than 10 ms.
        # The following code converts this output to a millisecond value.
        
        # remove the "real" part of the output
        TIME=${SPEED_TEST_OUTPUT#real }
        SECONDS=${TIME%%.*} # grab the digits before the decimal point
        SUB_SECOND=${TIME#*.}
        SUB_SECOND=${SUB_SECOND%[0-9]*} # grab the first digit after the decimal point

        if [ "$SECONDS" -gt "0" ] || [ "$SUB_SECOND" -gt "0" ]; then
            logerr "/dev/random is too slow (took $TIME seconds or more)"
            fail
        fi

        logok "/dev/random is sufficiently fast (took less than 100 milliseconds)"
    else
        logwarn "/dev/random not tested: one or more commands missing on the system. See above message(s) for details."
    fi
}


check_config_files() {
    loginfo "checking DEP config files..."
    
    if [ ! -e "$DANTE_JSON" ]; then
        logerr "dante.json not found in '$PATH_TO_CONFIG_FILES'"
        fail
    fi

    if [ -L "$CONFIG_JSON" ]; then
        # check for where the symlink points to
        TARGET="$PATH_TO_BUNDLE_DIR/$(readlink "$PATH_TO_BUNDLE_DIR/config.json")"
        TARGET=$(realpath "$TARGET")
        if ! [ -e "$TARGET" ]; then
            logerr "$TARGET does not exist at the symlink target pointed by '$PATH_TO_BUNDLE_DIR/config.json'."
            fail
        fi
    fi

    if ! [ -e "$CONFIG_JSON" ]; then
        logerr "config.json not found in '$PATH_TO_BUNDLE_DIR'."
        fail
    fi

    NUM_DEP_CORES=$(grep <"$DANTE_JSON" '"numDepCores"' | sed -E 's/[^[:digit:]]+([[:digit:]]+)[^[:digit:]]*/\1/')

    if [ -z "$NUM_DEP_CORES" ]; then
        logerr "'numDepCores' not set in $DANTE_JSON"
        fail
    fi

    logok "all config files found in $PATH_TO_BUNDLE_DIR"
}


check_network_interfaces() {
    loginfo "checking configured network interfaces..."

    interface_mode=$(grep -n interfaceMode "$DANTE_JSON" | cut -d: -f3 | sed -re 's/(\s|"|,)//g')

    # check that interfaceMode is set to one of the accepted values
    case $interface_mode in
        Direct | Switched)
            INTERFACE_MODE=$interface_mode
            ;;
        '')
            logerr "interfaceMode not specified in dante.json"
            ;;
        *)
            logerr "invalid interfaceMode in dante.json: '$interface_mode'. Possible values are: Direct, Switched"
            exit 1
            ;;
    esac

    interfaces=$(get_array_entries "interfaces")
    hardwareInterfaces=$(get_array_entries "hardwareInterfaces")

    # 'interfaces' can't contain the same interface twice (as opposed to
    # 'hardwareInterfaces' where that is valid)
    hardwareInterfaces=$(array_remove_duplicates "$hardwareInterfaces")
    duplicates=$(array_report_duplicates "$interfaces")
    if [ -n "$duplicates" ]; then
        logerr "the same network interface has been listed more than once in 'interfaces' in dante.json"
        exit 1
    fi

    # If 'enableHwTimestamping' is not found, hardware timestamping is disabled.
    # Otherwise, it must be one of true, false or "v1".
    ENABLE_HW_TIMESTAMPING=$(grep -w enableHwTimestamping ${DANTE_JSON} | cut -d: -f2 | sed -re 's/(\s|,)//g')
    if [ "${ENABLE_HW_TIMESTAMPING}" = true ] || [ "${ENABLE_HW_TIMESTAMPING}" = "\"v1\"" ]; then
        hardware_ts_in_use=1
    elif [ "${ENABLE_HW_TIMESTAMPING}" = false ] || [ "${ENABLE_HW_TIMESTAMPING}" = "" ]; then
        hardware_ts_in_use=0
    else
        logerr "invalid value for enableHwTimestamping in dante.json: '$ENABLE_HW_TIMESTAMPING'. Possible values are: true, false, \"v1\""
        exit 1
    fi

    # this is empty if 'dsaTaggedPackets' is not found, otherwise false or true
    DSA_TAGGED_PACKETS=$(grep -w dsaTaggedPackets ${DANTE_JSON} | cut -d: -f2 | sed -re 's/(\s|,)//g')

    # if hardware timestamping is in use and dsaTaggedPackets is true,
    # hardwareInterfaces _must_ be populated
    if [ "$hardware_ts_in_use" -eq 1 ] && [ "${DSA_TAGGED_PACKETS}" = true ] && [ -z "${hardwareInterfaces%%[[:space:]]*}" ]; then
        logerr "enableHwTimestamping is set to $ENABLE_HW_TIMESTAMPING, and dsaTaggedPackets is set to true, but hardwareInterfaces is empty/not found"
        exit 1
    fi

    all_ifaces="$interfaces"

    ifaces_no_spaces=$(echo "$all_ifaces" | tr -d '[:space:]')
    if [ "$ifaces_no_spaces" = "" ]; then
        logerr "no network interfaces found in dante.json"
        exit 1
    fi

    # global variable used by check_timestamping_config()
    INTERFACES="$interfaces"

    # if hardwareInterfaces is populated and hardware timestamping is in use,
    # we need to make sure entries in both interfaces and hardwareInterfaces
    # are found on the system
    # NOTE: make sure empty chars are filtered out
    if [ ! -z "${hardwareInterfaces%%[[:space:]]*}" ] && [ "$hardware_ts_in_use" -eq 1 ]; then
        all_ifaces="$all_ifaces"" ""$hardwareInterfaces"
        HW_INTERFACES="$hardwareInterfaces"
    fi

    for iface in ${all_ifaces}; do
        _=$(ethtool "$iface" 2>/dev/null)
        res="$?"
        if [ "$res" -ne 0 ]; then
            logerr "the interface \"$iface\" is specified in dante.json, but it was not detected on the system"
            exit 1
        else
            FOUND_IFACES="${FOUND_IFACES} ${iface}"
        fi
    done
}

check_network_speed() {

    friendly_speed() {
        num=$1
        case $num in
            10)
                echo "10 Mbps"
                ;;
            100)
                echo "100 Mbps"
                ;;
            1000)
                echo "1 Gbps"
                ;;
            10000)
                echo "10 Gbps"
                ;;
            *) # no match found, just echo the speed
                echo "$num"
                ;;
        esac
    }

    conf_speed_str=$(grep preferredLinkSpeed $DANTE_JSON | cut -d: -f2 | sed 's/[ "]*//g')

    case $conf_speed_str in
        LINK_SPEED_100M)
            conf_speed=100
            ;;
        LINK_SPEED_1G)
            conf_speed=1000
            ;;
        LINK_SPEED_10G)
            conf_speed=10000
            ;;
        '')
            conf_speed=1000  # default value for preferredLinkSpeed (LINK_SPEED_1G)
            ;;
        *)
            logerr "$conf_speed_str is not a valid value for preferredLinkSpeed"
            return
            ;;
    esac

    for iface in ${INTERFACES}; do
        neg_speed=$(ethtool $iface | grep -iw speed | awk '{print $2}' | tr -dc '0-9')

        # if the command failed we assume either the interface is not there or 
        # there were other issues that prevented the correct execution of ethtool
        res=$?
        if [ "$res" -ne 0 ]; then
            logwarn "could not find current negotiated speed for $iface"
        else
            # if neg_speed is empty, the interface is probably not up yet, so just ignore it
            if [ -z $neg_speed ]; then
                continue
            fi

            nspeed_friend=$(friendly_speed $neg_speed)
            cspeed_friend=$(friendly_speed $conf_speed)
            
            # if the negotiated speed is lower than what is set in dante.json, warn the user
            if [ $neg_speed -lt $conf_speed ]; then
                logwarn "$iface negotiated speed is ${nspeed_friend} while the configured preferred link speed is ${cspeed_friend}"
            else
                logok "$iface negotiated speed matches the preferred link speed (${cspeed_friend})"
            fi
        fi
    done
}


check_interface_coalescing() {
    # NOTE: ensure that this function is called _after_ invoking check_network_interfaces().
    #       This is because check_network_interfaces() is responsible for assigning the
    #       FOUND_IFACES variable, on which check_interface_coalescing() depends on.
    loginfo "check network interface(s) coalescing..."

    coa_check() {
        COA_ERR=0
        iface="$1"

        # test whether iface actually supports coalescing, return if false
        _=$(ethtool -c "$iface" 2>/dev/null)
        res="$?"

        if [ "$res" -ne 0 ]; then
            logwarn "network coalescing cannot be set/read on $iface"
            return
        fi

        rx_usec_val=$(ethtool -c "$iface" 2>/dev/null | grep "rx-usecs:" | cut -d' ' -f 2)
        tx_usec_val=$(ethtool -c "$iface" 2>/dev/null | grep "tx-usecs:" | cut -d' ' -f 2)

        if [ "$rx_usec_val" != 1 ]; then
            if [ "$rx_usec_val" != "n/a" ]; then
                logwarn "$iface has rx-usecs=$rx_usec_val, set to 1 for best performance"
                COA_ERR=1
            fi
        fi

        if [ "$tx_usec_val" != 1 ]; then
            if [ "$tx_usec_val" != "n/a" ]; then
                logwarn "$iface has tx-usecs=$tx_usec_val, set to 1 for best performance"
                COA_ERR=1
            fi
        fi

        if [ "$COA_ERR" -eq 0 ]; then
            logok "$iface coalescing settings ok"
        fi
    }

    if [ "$INTERFACE_MODE" = "Direct" ]; then 
        toparse=$INTERFACES
    elif [ "$INTERFACE_MODE" = "Switched" ]; then
        toparse=$HW_INTERFACES
    else
        logerr "unknown interfaceMode '$INTERFACE_MODE', coalescing check failed"
    fi        

    for iface in ${toparse}; do
        coa_check "${iface}"
    done
}


check_device_appears_in_config() {
    DEVICE=$1

    LS_OUTPUT=$(ls -l "/dev/$DEVICE")

    DEVICE_MAJOR_NUMBER=$(echo "$LS_OUTPUT" | grep -o "[0-9]\\+,") # grab the number before the comma
    DEVICE_MAJOR_NUMBER=${DEVICE_MAJOR_NUMBER%?}

    DEVICE_MINOR_NUMBER=$(echo "$LS_OUTPUT" | grep -o ",\\s\\+[0-9]\\+") # grab the number after the comma
    DEVICE_MINOR_NUMBER=${DEVICE_MINOR_NUMBER#?}

    DEVICE_PATH=$(grep < "$CONFIG_JSON" '"path"' | grep -c \"/dev/"$DEVICE"\")
    MAJOR_DEVICES=$(grep < "$CONFIG_JSON" '"major"' | grep -c "$DEVICE_MAJOR_NUMBER")

    if [ "$DEVICE_PATH" -eq 0 ] || [ "$MAJOR_DEVICES" -lt 2 ]; then
        RETVAL=0 #does not appear
    else
        RETVAL=1 #does appear
    fi
}


check_timestamping_config() {
    # NOTE: ensure that this function is called _after_ invoking check_network_interfaces().
    #       This is because check_network_interfaces() is responsible for assigning the
    #       FOUND_IFACES and ENABLE_HW_TIMESTAMPING variables, on which check_timestamping_config()
    #       depends.
    if [ "${ENABLE_HW_TIMESTAMPING}" = true ] || [ "${ENABLE_HW_TIMESTAMPING}" = "\"v1\"" ]; then # hardware timestamping being used
        loginfo "checking hardware timestamping config..."

        # Check if AES67 is being supported (if so, PTPv1 hardware timestamping cannot be used)
        aes67_supported=$(grep -w aes67Supported ${DANTE_JSON} | cut -d: -f2 | sed -re 's/(\s|,)//g')

        if [ "${aes67_supported}" = true ] && [ "${ENABLE_HW_TIMESTAMPING}" = "\"v1\"" ]; then
            logerr "$HW_TIMESTAMP_AES67_PTPV1_ERR_MSG"
            EXIT_VALUE=1
            return
        fi

        # if enableHwTimestamping is true, but dsaTaggedPackets is false, the NICs whose
        # HW timestamping capabilities need to be checked are those in $INTERFACES
        if [ "${DSA_TAGGED_PACKETS}" = true ]; then
            ifaces_to_check=$HW_INTERFACES
        else
            ifaces_to_check=$INTERFACES
        fi

        # For each configured interface, check its HW timestamping capabilities.
        #
        # We rely on the output of ethtool -T <interface>, which can vary across platforms/NICs.
        # A complete and detailed output would return:
        #
        # ...
        # Hardware Transmit Timestamp Modes:
        #          off                   (HWTSTAMP_TX_OFF)
        #          on                    (HWTSTAMP_TX_ON)
        # Hardware Receive Filter Modes:
        #          none                  (HWTSTAMP_FILTER_NONE)
        #          all                   (HWTSTAMP_FILTER_ALL)
        #
        # whereas in other instances the tool output can be:
        #
        # ...
        # Hardware Transmit Timestamp Modes:
        #          off
        #          on
        # Hardware Receive Filter Modes:
        #          none
        #          all
        #          ptpv1-l4-sync
        #          ptpv1-l4-delay-req
        #          ptpv2-l4-sync
        #          ptpv2-l4-delay-req
        #
        # which is why we can't parse for e.g. "HWTSTAMP_FILTER_ALL". Instead, we look 
        # for the corresponding keyword (in HWTSTAMP_FILTER_ALL case it'll be 'all') 
        # after the corresponding section (in HWTSTAMP_FILTER_ALL case after 
        # "Hardware Receive Filter Modes").
        #
        # Also:
        # check_network_interfaces() will check whether interfaceMode is 'switched' or 
        # 'direct', then populate FOUND_IFACES using the entries in 'interfaces' or 
        # 'hardwareInterfaces' respectively.

        all_capable_iface_count=0
        ptpv1_capable_iface_count=0
        ifaces_to_check_count=$(array_length "$ifaces_to_check")
        ptp_device_not_found=0

        for INTERFACE in ${ifaces_to_check}; do
            HW_TIMESTAMPING_INFO=$(ethtool -T "$INTERFACE" 2>/dev/null)

            transmit_filter='/Hardware Transmit Timestamp Modes/{flag=1;next} /Hardware Receive Filter Modes/{flag=0} flag'
            transmit_section=$(echo "$HW_TIMESTAMPING_INFO" | awk "${transmit_filter}")

            receive_filter='/Hardware Receive Filter Modes/{flag=1;next} flag'
            receive_section=$(echo "$HW_TIMESTAMPING_INFO" | awk "${receive_filter}")
            if ! echo "$transmit_section" | grep -q -w "on"; then
                logwarn "$INTERFACE does not support timestamping of TX packets (HWTSTAMP_TX_ON)"
            fi

            filter_all_supported=$(echo "$receive_section" | grep -c -w "all")
            if [ "$filter_all_supported" -eq 1 ]; then
                all_capable_iface_count=$((all_capable_iface_count+1))
            fi

            ptpv1_l4_event_supported=$(echo "$receive_section" | grep -c -w "ptpv1-l4-event")
            if [ "$ptpv1_l4_event_supported" -eq 1 ]; then
                ptpv1_capable_iface_count=$((ptpv1_capable_iface_count+1))
            fi

            if [ "${ENABLE_HW_TIMESTAMPING}" = true ]; then
                if [ "$filter_all_supported" -eq 0 ]; then
                    logwarn "$INTERFACE does not support timestamping of all RX packets (HWTSTAMP_FILTER_ALL)"
                fi
            else
                if [ "$ptpv1_l4_event_supported" -eq 0 ]; then
                    logwarn "$INTERFACE does not support timestamping of PTPv1 layer 4 event RX packets (HWTSTAMP_FILTER_PTPV1_L4_EVENT)"
                fi
            fi
            loginfo "checking interface $INTERFACE..."
 
            PTP_DEVICE=$(ls "/sys/class/net/$INTERFACE/device/ptp" 2>/dev/null)
            RETVAL=$?

            if [ $RETVAL -eq 0 ] && [ -n "$PTP_DEVICE" ]; then # at least one PTP device found, device names stored in variable
                logok "PTP device '$PTP_DEVICE' found for $INTERFACE interface"
                check_device_appears_in_config "$PTP_DEVICE"

                # ptp device for $INTERFACE found, but it has not been configured correctly in config.json
                # DEVICE_MAJOR_NUMBER is set by check_device_appears_in_config()
                if [ $RETVAL -eq 0 ]; then # not found
                    logwarn "'$PTP_DEVICE' has not been set up correctly in config.json"
                    print_device_config_message "$PTP_DEVICE" "$DEVICE_MAJOR_NUMBER"
                fi
            else
                logerr "'enableHwTimestamping' is set to $ENABLE_HW_TIMESTAMPING in dante.json but no PTP device has been found for $INTERFACE"
                logerr "$HW_TIMESTAMP_ERR_MSG"
                ptp_device_not_found=1
                EXIT_VALUE=1
            fi
        done

        # If enableHwTimestamping is true and:
        #
        # - Not all the interfaces support HWTSTAMP_FILTER_ALL, but
        # - They all support HWTSTAMP_FILTER_PTPV1_L4_EVENT, and
        # - AES67 is not supported
        #
        # Let the user know that setting it to "v1" instead is an option IF unicast 
        # domain clocking is not required.
        #
        # Similarly, if enableHwTimestamping has been set to "v1" and yet all the
        # interfaces support HWTSTAMP_FILTER_ALL (regardless of whether they all
        # support HWTSTAMP_FILTER_PTPV1_L4_EVENT), advise the user that it can be
        # set to true.
        #
        # NOTE: do NOT output these suggestions if the PTP device for any interface
        # is not found.
        if [ "$ptp_device_not_found" -eq 0 ]; then
            if [ "${ENABLE_HW_TIMESTAMPING}" = true ]; then
                if [ "$all_capable_iface_count" -ne "$ifaces_to_check_count" ] && [ "$ptpv1_capable_iface_count" -eq "$ifaces_to_check_count" ] && [ "${aes67_supported}" != true ]; then
                    loginfo "$HW_TIMESTAMP_PTPV1_OPTION_INFO_MSG"
                fi
            else
                if [ "$all_capable_iface_count" -eq "$ifaces_to_check_count" ]; then
                    loginfo "$HW_TIMESTAMP_ALL_OPTION_INFO_MSG"
                fi
            fi
        fi
    else
        loginfo "checking software timestamping config..."
        interfaces_count=0
        hw_supported_count=0

        for INTERFACE in ${INTERFACES}; do
            interfaces_count=$((interfaces_count+1))
            ERR=0

            TIMESTAMPING_INFO=$(ethtool -T "$INTERFACE" 2>/dev/null)
            if echo "$TIMESTAMPING_INFO" | grep -q "hardware-transmit" && echo "$TIMESTAMPING_INFO" | grep -q "hardware-receive"; then
                hw_supported_count=$((hw_supported_count+1))
            fi

            # check if the interface supports software-transmit, software-receive, and software-system-clock
            if ! echo "$TIMESTAMPING_INFO" | grep -q "software-transmit"; then
                logwarn "$INTERFACE does not support software-transmit (SOF_TIMESTAMPING_TX_SOFTWARE)"
                ERR=1
            fi

            if ! echo "$TIMESTAMPING_INFO" | grep -q "software-receive"; then
                logwarn "$INTERFACE does not support software-receive (SOF_TIMESTAMPING_RX_SOFTWARE)"
                ERR=1
            fi

            if ! echo "$TIMESTAMPING_INFO" | grep -q "software-system-clock"; then
                logwarn "$INTERFACE does not support software-system-clock (SOF_TIMESTAMPING_SOFTWARE)"
                ERR=1
            fi

            if [ 0 -eq "${ERR}" ]; then
                logok "all software timestamping flags found for $INTERFACE"
            fi
        done

        # if all interfaces found in $INTERFACES support HW timestamping,
        # suggest the user to enable it
        if [ ${interfaces_count} -eq ${hw_supported_count} ]; then
            for INTERFACE in ${INTERFACES}; do
                logwarn "$INTERFACE supports hardware timestamping"
            done    
            print_timestamp_suggestion
        fi
    fi
}


check_hw_clock_config() {
    loginfo "checking hardware clock config..."

    USE_HW_CLOCK=$(grep < "$DANTE_JSON" '"useHwClock"' | grep -c true)

    if [ "$USE_HW_CLOCK" -eq 1 ]; then # hardware clock enabled
        if [ -c /dev/extclkin ]; then # /dev/extclkin found
            check_device_appears_in_config "extclkin" # also sets a variable $DEVICE_MAJOR_NUMBER

            if [ $RETVAL -eq 0 ]; then
                logwarn "'useHwClock' is set in dante.json and /dev/extclkin exists, however it has not been configured in config.json"
                print_device_config_message "extclkin" "$DEVICE_MAJOR_NUMBER"
            else
                logok "/dev/extclkin set as hardware clock. Ensure proper function by verifying the minor device number in config.json"
            fi
        else
            logerr "'useHwClock' has been set to true in dante.json but /dev/extclkin has not been found on the system. Please disable useHwClock."
            EXIT_VALUE=1
        fi
    else
        logok "hardware clock disabled in dante.json, nothing to do"
    fi
}


# Given a comma-separated list of CPU cores, including ranges (entries separated by a dash),
# return an array containing all the cores - i.e. all the cores separated by spaces.
# NOTE: the returned array must not have any leading or trailing spaces
create_cpu_array() {
    # Convert to space separated instead of comma separated
    CORES_LIST=$(echo "$1" | sed 's/,/ /g')

    ALL_CORES=""
    for CORE in $CORES_LIST
    do
        # Is it a ranged value?
        RANGE_END=$(echo "$CORE" | cut -f2 -d -)
        if [ -n "$RANGE_END" ]; then
            RANGE_START=$(echo "$CORE" | cut -f1 -d -)
            if [ "$RANGE_START" -lt "$RANGE_END" ]; then
                CORES=$(seq "$RANGE_START" "$RANGE_END")
            else
                CORES=$(seq "$RANGE_END" "$RANGE_START")
            fi
            CORES=$(echo "$CORES" | tr '\n' ' ')
            CORES=${CORES%?}
        else
            CORES="$CORE"
        fi

        if [ "$ALL_CORES" = "" ]; then
            ALL_CORES="$CORES"
        else
            ALL_CORES="$ALL_CORES $CORES"
        fi
    done

    echo "$ALL_CORES"
}


# Given the arguments to depconfig, create an array of CPU cores for a specific cores flag list.
# Assign the array to the variable CPU_CORES.
#
# Report and remove any duplicate entries. CPU_CORES is empty if an error occurs.
create_depconfig_cpu_array() {
    DEPCONFIG_ARGS=$1
    CORES_FLAG=$2
    CPU_CORES=""

    CORES_ARG=$(echo "$DEPCONFIG_ARGS" | sed -E "s/.+\"--?$CORES_FLAG\", \"([[:digit:],-]+)\".+/\1/") # Note: hyphen must be the last char in the range
    if [ "$CORES_ARG" = "$DEPCONFIG_ARGS" ]   # sed failed to find a match 
    then
        logerr "$CORES_FLAG value is missing/invalid"
    else
        CPU_CORES=$(create_cpu_array "$CORES_ARG")

        DUPLICATES=$(array_report_duplicates "$CPU_CORES")
        if [ -n "$DUPLICATES" ]; then
            logwarn "the following CPU(s) are duplicated in $CORES_FLAG: $DUPLICATES"
            CPU_CORES=$(array_remove_duplicates "$CPU_CORES")
        fi

        COUNT=$(array_length "$CPU_CORES")
        if [ "$COUNT" -gt "$NUM_CORES" ]; then
            logerr "CORES_FLAG specifies more CPUs than are available on this system"
            CPU_CORES=""
        fi

        for CORE in $CPU_CORES; do
            if [ "$CORE" -gt "$MAX_CORE_ID" ]; then
                logerr "$CORES_FLAG contains the invalid core $CORE"
                CPU_CORES=""
                return
            fi
        done
    fi
}


check_cpuset_config() {
    loginfo "checking cpuset config..."

    CPUSET_CONFIG_OK="true"

    # if numDepCores != 0 and one or more cgroup mount is missing we can't perform these check reliably
    if [ "$NUM_DEP_CORES" -ne 0 ] && [ "$CGROUP_ERR" -eq 1 ]; then
        logwarn "cpuset config check skipped: one or more cgroup mount is missing"
        return
    fi

    if [ "$NUM_DEP_CORES" -ne 0 ]; then
        loginfo "DEP is configured to use $NUM_DEP_CORES core(s)"

        if [ "$NUM_DEP_CORES" -gt "$NUM_CORES" ]; then
            logerr "'numDepCores' exceeds total CPU core count ($NUM_DEP_CORES vs $NUM_CORES)"
            CPUSET_CONFIG_OK="false"
        fi

        # Not a real while loop - it just facilitates neater control transfer
        while [ "$CPUSET_CONFIG_OK" = "true" ]; do
            # Account for any offline cores...
            OFFLINE_CPUS=$(cat /sys/devices/system/cpu/offline)
            if [ -n "$OFFLINE_CPUS" ]; then
                OFFLINE_CPU_CORES=$(create_cpu_array "$OFFLINE_CPUS")
            else
                OFFLINE_CPU_CORES=""
            fi

            # Get DEP and non-DEP core specs from config, using defaults if not specified
            DEPCONFIG_DEP_CORES_FLAG="dep-cpu-cores"
            DEPCONFIG_OTHER_CORES_FLAG="other-cpu-cores"
            DEPCONFIG_ARGS=$(grep <"$CONFIG_JSON" "$DEPCONFIG_DEP_CORES_FLAG""\|""$DEPCONFIG_OTHER_CORES_FLAG")

            # ***** Check DEP cores *****

            DEPCONFIG_DEP_CORES_SPECIFIED=$(echo "$DEPCONFIG_ARGS" | grep "$DEPCONFIG_DEP_CORES_FLAG")
            if [ -n "$DEPCONFIG_DEP_CORES_SPECIFIED" ]; then
                create_depconfig_cpu_array "$DEPCONFIG_ARGS" "$DEPCONFIG_DEP_CORES_FLAG"
                DEP_CPU_CORES="$CPU_CORES"
                if [ "$DEP_CPU_CORES" = "" ]; then
                    CPUSET_CONFIG_OK="false"
                    break
                fi
                DEP_BY_DEFAULT=""
            else
                # Default to first numDepCores cores starting from 0
                DEFAULT_CORES=$(seq 0 $((NUM_DEP_CORES-1)))
                DEP_CPU_CORES=$(echo "$DEFAULT_CORES" | tr '\n' ' ')
                DEP_CPU_CORES=${DEP_CPU_CORES%?}
                DEP_BY_DEFAULT="by default"
            fi
            DEP_CPU_CORES_LIST=$(echo "$DEP_CPU_CORES" | tr ' ' ',')
            loginfo "DEP configured to use core(s) $DEP_CPU_CORES_LIST $DEP_BY_DEFAULT"

            OFFLINE_DEP_CPU_CORES=$(array_report_overlap "$DEP_CPU_CORES" "$OFFLINE_CPU_CORES")
            if [ -n "$OFFLINE_DEP_CPU_CORES" ]; then
                OFFLINE_DEP_CPU_CORES_LIST=$(echo "$OFFLINE_DEP_CPU_CORES" | tr ' ' ',')
                logwarn "the following core(s) are offline: $OFFLINE_DEP_CPU_CORES_LIST"
                DEP_CPU_CORES=$(array_remove_entries "$DEP_CPU_CORES" "$OFFLINE_CPU_CORES")
            fi

            DEP_CPU_CORE_COUNT=$(array_length "$DEP_CPU_CORES")
            if [ "$DEP_CPU_CORE_COUNT" -gt "$NUM_DEP_CORES" ]; then    # Can only be true if dep-cpu-cores specified
                DEP_CPU_CORES=$(array_trim_end "$DEP_CPU_CORES" $((DEP_CPU_CORE_COUNT-NUM_DEP_CORES)))
                DEP_CPU_CORES_LIST=$(echo "$DEP_CPU_CORES" | tr ' ' ',')
                loginfo "ignoring surplus core(s) in $DEPCONFIG_DEP_CORES_FLAG - using $DEP_CPU_CORES_LIST only"
            fi

            DEP_CPU_CORE_COUNT=$(array_length "$DEP_CPU_CORES")
            if [ "$DEP_CPU_CORE_COUNT" -lt "$NUM_DEP_CORES" ]; then
                logerr "insufficient number of cores allocated/available for DEP"
                CPUSET_CONFIG_OK="false"
                break
            fi

            # ***** Check 'other' cores *****

            DEPCONFIG_OTHER_CORES_SPECIFIED=$(echo "$DEPCONFIG_ARGS" | grep "$DEPCONFIG_OTHER_CORES_FLAG")
            if [ -n "$DEPCONFIG_OTHER_CORES_SPECIFIED" ]; then
                create_depconfig_cpu_array "$DEPCONFIG_ARGS" "$DEPCONFIG_OTHER_CORES_FLAG"
                OTHER_CPU_CORES="$CPU_CORES"
                if [ "$OTHER_CPU_CORES" = "" ]; then
                    CPUSET_CONFIG_OK="false"
                    break
                fi
                OTHER_BY_DEFAULT=""
            else
                # Default to all system cores
                OTHER_CPU_CORES=$(create_cpu_array "$(cat /sys/devices/system/cpu/possible)")
                OTHER_BY_DEFAULT="by default"
            fi

            # Check whether DEP is using its cores exclusively - if so, remove them from the non-DEP list
            PERCENT_CPU_SHARE=$(grep <"$DANTE_JSON" '"percentCpuShare"' | sed -E 's/[^[:digit:]]+([[:digit:]]+)[^[:digit:]]*/\1/')
            if [ -n "$PERCENT_CPU_SHARE" ] && [ "$PERCENT_CPU_SHARE" -lt 100 ]; then # cpu sharing enabled
                loginfo "DEP not in exclusive CPU mode ('percentCpuShare' option specified and not set to 100)"

                ls /sys/fs/cgroup/cpu/cpu.shares > /dev/null 2> /dev/null
                RETVAL=$?

                if [ $RETVAL -ne 0 ]; then # cpu share controller not found
                    logerr "the 'percentCpuShare' option is being used in dante.json but the controller /sys/fs/cgroup/cpu/cpu.shares has not been found on this system"
                    CPUSET_CONFIG_OK="false"
                    break
                fi
            else
                loginfo "DEP in exclusive CPU mode ('percentCpuShare' option not specified, or set to 100)"

                OTHER_CPU_CORES_WITH_ANY_OVERLAP="$OTHER_CPU_CORES"
                OTHER_CPU_CORES=$(array_remove_entries "$OTHER_CPU_CORES" "$DEP_CPU_CORES")

                # Make sure we have at least one "other" core
                if [ -z "$OTHER_CPU_CORES" ]; then
                    OTHER_CPU_CORES_WITH_ANY_OVERLAP_LIST=$(echo "$OTHER_CPU_CORES_WITH_ANY_OVERLAP" | tr ' ' ',')
                    logerr "other (non-DEP) core(s) are: $OTHER_CPU_CORES_WITH_ANY_OVERLAP_LIST $OTHER_BY_DEFAULT"
                    logerr "DEP is using all of these cores and the 'percentCpuShare' option is not set. Please set this option or assign different cores."
                    CPUSET_CONFIG_OK="false"
                    break
                fi
            fi

            OTHER_CPU_CORES_LIST=$(echo "$OTHER_CPU_CORES" | tr ' ' ',')
            loginfo "other (non-DEP) core(s) configured as $OTHER_CPU_CORES_LIST $OTHER_BY_DEFAULT"

            OFFLINE_OTHER_CPU_CORES=$(array_report_overlap "$OTHER_CPU_CORES" "$OFFLINE_CPU_CORES")
            if [ -n "$OFFLINE_OTHER_CPU_CORES" ]; then
                OFFLINE_OTHER_CPU_CORES_LIST=$(echo "$OFFLINE_OTHER_CPU_CORES" | tr ' ' ',')
                logwarn "the following core(s) are offline: $OFFLINE_OTHER_CPU_CORES_LIST"
                OTHER_CPU_CORES=$(array_remove_entries "$OTHER_CPU_CORES" "$OFFLINE_CPU_CORES")
            fi

            OTHER_CPU_CORE_COUNT=$(array_length "$OTHER_CPU_CORES")
            if [ "$OTHER_CPU_CORE_COUNT" -eq 0 ]; then
                logerr "no non-DEP core(s) available!"
                CPUSET_CONFIG_OK="false"
                break
            fi

            # **** Check whether any services are using the same cores as DEP ****

            loginfo "checking whether DEP cpuset clashes with any services... "
            CPUSETS=$(ls /sys/fs/cgroup/cpuset/*/cpuset.cpus 2>/dev/null)
            CLASH_FOUND="false"
            for CPUSET in $CPUSETS
            do
                SERVICE_NAME=$(echo "$CPUSET" | cut -d / -f6)
                if [ "$SERVICE_NAME" != dante ]; then
                    SERVICE_CPUS=$(tr <"/sys/fs/cgroup/cpuset/$SERVICE_NAME/cpuset.cpus" , \\n)
                    SERVICE_CPU_CORES=$(create_cpu_array "$SERVICE_CPUS")
                    OVERLAPPING=$(array_report_overlap "$SERVICE_CPU_CORES" "$DEP_CPU_CORES")
                    if [ -n "$OVERLAPPING" ]; then
                        if [ "$CLASH_FOUND" = "false" ]; then
                            CLASH_FOUND="true"
                        fi
                        logerr "service '$SERVICE_NAME' is trying to use the following DEP core(s): $OVERLAPPING"
                        CPUSET_CONFIG_OK="false"
                    fi
                fi
            done

            if [ "$CLASH_FOUND" = "true" ]; then
                logerr "service cpuset clashes with DEP were found"
                logerr "if DEP and/or service cores cannot be modified, you may need to disable these services and reboot"
            else
                logok "no clashes found"
            fi

            # Checks finished
            break
        done

        if [ $CPUSET_CONFIG_OK = "true" ]; then
            logok "cpuset config ok"
        else
            EXIT_VALUE=1
        fi
    else
        logok "'numDepCores' is set to 0, nothing to do"
    fi
}


check_cpu_performance_scaling() {
    loginfo "checking CPU frequency scaling settings..."

    if [ -n "$KERNEL_CONFIG" ]; then
        CPU_FREQ=$(echo "$KERNEL_CONFIG" | grep -c "CONFIG_CPU_FREQ=y")
        CPU_FREQ_MODULE=$(echo "$KERNEL_CONFIG" | grep -c "CONFIG_CPU_FREQ=m")
        if [ "$CPU_FREQ" -eq 1 ] || [ "$CPU_FREQ_MODULE" -eq 1 ]; then
            loginfo "kernel supports CPU frequency scaling"

            # Check whether the "performance" scaling governor is supported
            RETVAL=$(echo "$KERNEL_CONFIG" | grep -c "CONFIG_CPU_FREQ_GOV_PERFORMANCE is not set")
            if [ "$RETVAL" -eq 1 ]; then
                logwarn "'CONFIG_CPU_FREQ_GOV_PERFORMANCE' not set in kernel config. Please rebuild the kernel with this option set to either 'y' or 'm'."
                return
            fi

            RETVAL=$(echo "$KERNEL_CONFIG" | grep -c "CONFIG_CPU_FREQ_GOV_PERFORMANCE=n")
            if [ "$RETVAL" -eq 1 ]; then
                logwarn "'CONFIG_CPU_FREQ_GOV_PERFORMANCE' is set to 'n'. Please rebuild the kernel with this option set to either 'y' or 'm'."
                return
            fi

            logok "kernel supports performance scaling governor"

            # For each CPU, check that:
            # 1) The scaling governor in use is 'performance'
            # 2) scaling_max_freq is equal to cpuinfo_max_freq

            ALL_CORES=$(seq 0 "$MAX_CORE_ID")
            for CORE in $ALL_CORES; do
                SYSFS_CPU_FREQ_PATH="/sys/devices/system/cpu/cpu${CORE}/cpufreq"

                SCALING_GOVERNOR=$(cat "${SYSFS_CPU_FREQ_PATH}/scaling_governor")
                if [ "$SCALING_GOVERNOR" != "performance" ]; then
                    logwarn "cpu$CORE scaling_governor is set to '$SCALING_GOVERNOR' instead of 'performance'"
                    continue
                fi

                SCALING_MAX_FREQ=$(cat "${SYSFS_CPU_FREQ_PATH}/scaling_max_freq")
                CPUINFO_MAX_FREQ=$(cat "${SYSFS_CPU_FREQ_PATH}/cpuinfo_max_freq")
                if [ "$SCALING_MAX_FREQ" -lt "$CPUINFO_MAX_FREQ" ]; then
                    logwarn "cpu$CORE scaling_max_freq is less than cpuinfo_max_freq"
                fi

                logok "cpu$CORE scaling_governor ok (performance, scaling_max_req=$SCALING_MAX_FREQ, cpuinfo_max_req=$CPUINFO_MAX_FREQ)"
            done
        else
            logok "kernel does not support CPU frequency scaling"
        fi
    else
        logwarn "unable to check due to missing kernel config"
    fi
}


# Encoding check routines

# Convert hex characters used in custom encoding values to uppercase, for simplified checking
convert_encoding_to_uppercase()
{
    ENCODING=$1
    UPPERCASE_ENCODING=$(echo "$ENCODING" | tr '[a-f]' '[A-F]')

    echo "$UPPERCASE_ENCODING"
}

# Check if an encoding is valid
# ENCODING_VALID will be set as follows:
#
# To 1 if the encoding is PCM
# To 2 if the encoding is custom (value between 0x1300 and 0x13FF)
# To 0 otherwise (i.e. invalid)
check_encoding_valid()
{
    ENCODING=$1

    CUSTOM_ENCODING_VAL=$(echo $(($ENCODING)))
    CUSTOM_ENCODING_PREFIXED=$(echo "$ENCODING" | grep -c "0x13")
    if [ $CUSTOM_ENCODING_PREFIXED -eq 1 ] && [ $CUSTOM_ENCODING_VAL -ge 4864 ] && [ $CUSTOM_ENCODING_VAL -le 5119 ]; then
        ENCODING_VALID=2
        return
    fi

    if [ "$ENCODING" != "PCM16" ] && [ "$ENCODING" != "PCM24" ] && [ "$ENCODING" != "PCM32" ]; then
        ENCODING_VALID=0
        return
    fi
    ENCODING_VALID=1
}

check_encodings()
{
    loginfo "checking encoding settings... "

    DEFAULT_ENCODING=$(grep -n defaultEncoding "$DANTE_JSON" | cut -d: -f3 | sed -re 's/(\s|"|,)//g')

    if [ -n "$DEFAULT_ENCODING" ]; then
        check_encoding_valid "$DEFAULT_ENCODING"
        if [ $ENCODING_VALID -eq 0 ]; then
            logerr "defaultEncoding $ENCODING is invalid"
            EXIT_VALUE=1
            return
        elif [ "$ENCODING_VALID" -eq 2 ]; then
            DEFAULT_ENCODING=$(convert_encoding_to_uppercase "$DEFAULT_ENCODING")
        fi
    fi

    SUPPORTED_ENCODINGS_PRESENT=$(grep -n  supportedEncodings "$DANTE_JSON" | cut -d: -f1)
    if [ -z "$SUPPORTED_ENCODINGS_PRESENT" ]; then
        if [ -n "$DEFAULT_ENCODING" ]; then
            logok "supportedEncodings not specified, using only defaultEncoding"
        else
            logwarn "neither supportedEncodings nor defaultEncoding specified - PCM24 will be used by default"
        fi
        return
    fi

    SUPPORTED_ENCODINGS=$(get_array_entries "supportedEncodings")

    NUM_SUPPORTED_ENCODINGS=$(array_length "$SUPPORTED_ENCODINGS")
    if [ "$NUM_SUPPORTED_ENCODINGS" -eq 0 ]; then
        logerr "supportedEncodings cannot be specified as empty"
        EXIT_VALUE=1
        return
    fi

    DUPLICATES=$(array_report_duplicates "$SUPPORTED_ENCODINGS")
    if [ -n "$DUPLICATES" ]; then
        logerr "supportedEncodings contains the duplicate encoding(s) $DUPLICATES"
        EXIT_VALUE=1
        return
    fi

    DEFAULT_ENCODING_FOUND=0
    CUSTOM_ENCODING_COUNT=0
    for ENCODING in $SUPPORTED_ENCODINGS; do
        check_encoding_valid "$ENCODING"
        if [ "$ENCODING_VALID" -eq 0 ]; then
            logerr "supportedEncodings contains the invalid encoding $ENCODING"
            EXIT_VALUE=1
            return
        elif [ "$ENCODING_VALID" -eq 2 ]; then
            CUSTOM_ENCODING_COUNT=$((CUSTOM_ENCODING_COUNT+1))
            if [ $CUSTOM_ENCODING_COUNT -gt 4 ]; then
                logerr "supportedEncodings cannot contain more than 4 custom encodings"
                EXIT_VALUE=1
                return
            fi
            ENCODING=$(convert_encoding_to_uppercase "$ENCODING")
        fi

        if [ "$ENCODING" = "$DEFAULT_ENCODING" ]; then
            DEFAULT_ENCODING_FOUND=1
        fi
    done

    if [ -n "$DEFAULT_ENCODING" ]; then
        if [ "$DEFAULT_ENCODING_FOUND" -eq 0 ]; then
            logerr "supportedEncodings does not contain the specified defaultEncoding"
            EXIT_VALUE=1
            return
        fi
        logok "encoding settings ok"
    else
        logok "defaultEncoding not specified, the default encoding used will be the first entry in supportedEncodings"
    fi
}

check_device_entries() {
    entries="i2cBus extClockInputDev"

    for entry in $entries; do
        # check whether the entry is found - if yes, check its value
        entry_value=$(grep "$entry" "$DANTE_JSON" | awk '{ print $2 }' | sed 's/[",]//g')

        if [ -n "$entry_value" ]; then
            starts_with "$entry_value" "/dev"  
            parse_res="$?"
            if [ $parse_res -ne 0 ]; then
                logerr "value for '$entry' in $DANTE_JSON doesn't start with '/dev'"
                EXIT_VALUE=1
            fi
        fi
    done
}

check_minimum_glibc_version() {
    print_container_warning() {
        echo "            if unable to start DEP due to crun issues, consider compiling crun"
        echo "            (or an alternative OCI-compliant container runtime) using a compatible"
        echo "            toolchain for this system"
    }
    # The minimum glibc version needed by crun is obtained as shown below:
    #
    #   $ objdump -T crun | grep GLIBC | sed 's/.*GLIBC_\([.0-9]*\).*/\1/g' | sort -Vu | tail -n1
    #     2.14
    #
    REQUIRED_GLIBC=2.14

    # determine installed glibc version
    if command -v ldd >/dev/null 2>&1; then
        INSTALLED_GLIBC=$(ldd --version 2>/dev/null | awk 'NR==1{print $NF}')
    elif [ -x /lib/libc.so.6 ]; then
        INSTALLED_GLIBC=$(/lib/libc.so.6 2>/dev/null | awk 'NR==1{print $NF}')
    elif [ -x /lib64/libc.so.6 ]; then
        INSTALLED_GLIBC=$(/lib64/libc.so.6 2>/dev/null | awk 'NR==1{print $NF}')
    else
        logwarn "could not determine installed glibc version:"
        print_container_warning
        return
    fi

    REQUIRED_MAJOR=$(echo "$REQUIRED_GLIBC" | cut -d'.' -f1 | tr -dc '0-9')
    REQUIRED_MINOR=$(echo "$REQUIRED_GLIBC" | cut -d'.' -f2 | tr -dc '0-9')

    INSTALLED_MAJOR=$(echo "$INSTALLED_GLIBC" | cut -d'.' -f1 | tr -dc '0-9')
    INSTALLED_MINOR=$(echo "$INSTALLED_GLIBC" | cut -d'.' -f2 | tr -dc '0-9')

    if [ "$INSTALLED_MAJOR" -lt "$REQUIRED_MAJOR" ] || { [ "$INSTALLED_MAJOR" -eq "$REQUIRED_MAJOR" ] && [ "$INSTALLED_MINOR" -lt "$REQUIRED_MINOR" ]; }; then
        logwarn "installed glibc ($INSTALLED_GLIBC) is older than required ($REQUIRED_GLIBC) for container crun"
        print_container_warning
        return
    fi

    logok "installed glibc ($INSTALLED_GLIBC) is compatible with container crun"
}

check_minimum_glibc_version
check_architecture
check_config_files
check_kernel_config
check_cgroups
check_rng_speed

# NOTE: check_network_interfaces must preceed check_interface_coalescing()
#       and check_timestamping_config() invocations because it assigns the 
#       FOUND_IFACES variable on which both functions depend on.
check_network_interfaces
check_network_speed
check_interface_coalescing
check_timestamping_config
check_hw_clock_config
check_cpuset_config
check_cpu_performance_scaling
check_encodings
check_device_entries

exit $EXIT_VALUE

#
# Copyright © 2022-2025 Audinate Pty Ltd ACN 120 828 006 (Audinate). All rights reserved.
#
